-------------------------------------------------------------------------------
--   
--  File:		rolloutCreator.ms
--  Description:	Localization friendly helper struct for dynamically creating rollouts
--  By:		Ravi Karra [Discreet] 			ravi.karra@discreet.com
--	 
--  Version:	1.01
--  Version:	1.02 - Larry Minton [Discreet] 
--						changed <string1> += <string2> to append string1 string2
--						added addText method
--  Version:	1.03 - Larry Minton [Autodesk] 
--						changed string additions to string appends, which is much more efficient
--						since many intermediate strings are not created that then need to be
--						garbage collected
--  Declarations:
/*
	rolloutCreator <rollout_name> <rollout_caption> [width:] [height:]
		creates an instance of rolloutCreator, assign it to a variable
		width - width of the rollout/dialog to be created
		height - of the rollout/dialog to be created
		
		eg:
	 	rci = rolloutCreator "myRollout" "My Rollout"
		

	.begin()
		this function needs to be called immediately after the instance is created, this does the initialization

	.addLocal <local_name> [init:]
		<local_name> 
		name of the local
		
		[init:]
		what the local should be initialized to
	
	.addControl <control_type> <control_name> <control_caption> [paramStr:<string>] =
		adds a control to the rollout
		
		<control_type>
		can be any of named rolloutControls eg: #button, #spinner, #activeXControl etc
		
		<control_name>
		variable name of the control by which it is referred eg: #btnButton
		
		<control_caption>
		caption of the control "My Button"
		
		[paramStr:]
		an optional string representation of all the keyword parameters that needs to be passed to the control
		eg: "width:100 height:20 align:#right"
		
		eg:
		rci.addControl #button #myButton "My Button"
	
	.addHandler <control_name> <event_type> [paramStr:<string>]  [codeStr:<string>] [filter:<boolean>]
		adds an event handler for the controls previously added 
		
		<control_name>
		the variable passed during the control creation
		 
		<event_type>
		any of the events supported by the control, eg: #changed, #pressed, #selected
		
		[paramStr:<string>]
		an optional string representation of all the positional and keyword parameters that are passed to the event	

		[codeStr:<string>]
		a string representation of the event handler code, if the string contains sub-strings, enclose them in two character '@' 
		and pass on\true for the filter: parameter

		[filter:<boolean>]
		if true, converts '@' to quote in codeStr
				
		eg:
		rci.addHandler #myButton #pressed codeStr:"MessageBox @Hey@" filter:on
		will add an event handler for button named "myButton". When the button is clicked, messagebox pops up with text "hey" in it.
		
	.addText <string> [filter:<boolean>]
		adds string to rollout definition. Typically used for function definitions.

		[filter:<boolean>]
		if true, converts '@' to quote in string
				
	.end()
		this function has to be called whenever all the required control and their event handler's are called. This function forms 
		the rollout string, evaluates it and returns the definition which can passed to createDialog and addRollout functions.
	
	Complete Example:
		 rci = rolloutCreator "myRollout" "My Rollout"
		 rci.begin()
			rci.addControl #button #myButton "My Button"
			rci.addHandler #myButton #pressed filter:on codeStr:"MessageBox @Isn't this cool@ title:@Wow@"
		createDialog (rci.end())
*/
-------------------------------------------------------------------------------

global __rcCounter
if __rcCounter == undefined then __rcCounter = 0
struct rolloutCreator
(
	-- variables
public
	name, 
	caption = "", 
	str, 
	def, 
	width, 
	height, 
	sourceDefinitionScriptFilename = "",
private
	quote = "\"", 
	cr = "\n", 
	space = " ", 
	tab = "\t", 
	equal = " = ", 
	localStr = "\tlocal ", 
	atStr = "@", 
	onStr = "\ton ", 
	doStr = "do \n\t(\n", 
	doEndStr = "\n\t)\n", 
	rollloutStr = "rollout ",
	widthStr = " width:", 
	heightStr = " height:", 
	beginParenStr = "\n(\n", 
	endParenStr = ")\n",
	defName = "rolloutCreator",
	
	-- functions
public
	fn begin = 
	( 
		if name == undefined then
		(
			__rcCounter += 1
			name = defName + __rcCounter as string
		)
		str = ""
	),
	
	fn addLocal name init: = 
	(
		append str localStr
		append str (name as string)
		if init != unsupplied then 
		(
			append str equal
			append str (init as string)
		)
		append str cr
	),

	fn addControl type name caption paramStr:"" =
	(	
		append str tab
		append str (type as string)
		append str space
		append str (name as string)
		append str space
		append str quote 
		append str caption
		append str quote
		append str space
		append str paramStr
		append str cr
	),
	
	fn strFilter codeStr =
	(
		local first_is_at = codeStr[1] == atStr
		local last_is_at = codeStr[codeStr.count] == atStr
		local fltStr = filterString codeStr atStr
		local resCodeStr = ""
		if first_is_at then append resCodeStr quote
		append resCodeStr fltStr[1]
		for i=2 to fltStr.count do
		(
			append resCodeStr quote
			append resCodeStr fltStr[i]
		)
		if last_is_at then append resCodeStr quote
		resCodeStr -- return value
	),

	fn addHandler ctrl event paramStr:"" filter:on codeStr:"" =
	(
		if filter do codeStr = (strFilter codeStr)
		append str onStr
		append str (ctrl as string)
		append str space
		append str (event as string)
		append str space
		append str (paramStr as string)
		append str space
		append str doStr
		append str tab
		append str tab
		append str codeStr
		append str doEndStr
	),
	
	fn addText txt filter:on =
	(
		if filter do txt = (strFilter txt )
		append str tab
		append str txt 
		append str cr
	),
	
	fn end = 
	( 
		local dStr = ""
		append dStr rollloutStr
		append dStr (name as string)
		append dStr space
		append dStr quote
		append dStr caption
		append dStr quote
		if width != undefined then
		(
 			append dStr widthStr
			append dStr (width as string)
		)
		if height != undefined then
		(
			append dStr heightStr
			append dStr (height as string)
		)
		append dStr beginParenStr
		append dStr str
		append dStr endParenStr
		free str -- release memory held by string
		str = dStr
		local ss = stringstream str
		ss.filename = sourceDefinitionScriptFilename -- used for resolving resource ids
		-- print str
		def = execute ss
		free ss -- release memory held by stringstream
		def -- return value
	),
	
	-- This function is used to specify the filename of the script providing the rollout source definition.
	-- By setting the filename property on the StringStream that is executed, any resource ids are resolved
	-- by loading the resource file for the source definition script
	fn setSourceDefinitionScriptFilename scriptFilename =
	(
		sourceDefinitionScriptFilename = scriptFilename
	)
)

/*-- Usage
-- Create an instance of the rolloutCreator passing the name and the caption
rfTest = rolloutCreator "rfTestN" "rfTestC" --width:300 height:100
-- Start creating the rollout
rfTest.begin()
	rfTest.addLocal "L1"
	rfTest.addLocal "L2" init:20
	rfTest.addControl #button #myButton "My Button"			-- add a button
	rfTest.addHandler #myButton #pressed filter:on codeStr:"MessageBox @Look to the \@Light\@ thing@" 
	rfTest.addText "-- this is @test text line 1@" filter:false
	rfTest.addText "-- this is @test text line 2@" filter:true
rfTest.end()
createDialog rfTest.def
*/
-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQArmLGGkH/vp2M/Rprrp7toTpR18XnwVQRt
-- Msp5v+tqxRPmOGhY9hQgzS3bZz+8L9TjunRnWkdJIWND7L4G9JNFKaVcr2nejFmc
-- +oSogQXg3okFqonCVi3O97e8Ap0S6SotuTKSHIdTnfiCB9uQoYnky2EcFGOVfBTn
-- 7KMh5CzxcNgRhqfrOUrnsxMgITPYXAjZNOGwcCHQURtJgITSsBGA+utyXCfmtiXT
-- GNpW+tpaUsdoeL9qmpRJ1cV+mPN4ZB/Xk1ZZ/ROxDWZju85ZHBFjt7cYN9VsVjJJ
-- AlmobAetHjFuN6+yAJ0ixImEqDbNjX7WtXtjRv/YlfavMSWaxgOl
-- -----END-SIGNATURE-----